/*
 * Copyright (C) 2012 Sergey Matyukevich <geomatsi@gmail.com>
 * Copyright (C) 2013 Dmitry Eremin <dmitry_eremin@mentor.com>
 * Copyright (C) 2013 Mentor Graphics
 * Copyright (C) 2015 Advanced Driver Information Technology Joint Venture GmbH
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting documentation, and
 * that the name of the copyright holders not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  The copyright holders make no representations
 * about the suitability of this software for any purpose.  It is provided "as
 * is" without express or implied warranty.
 *
 * THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
 * OF THIS SOFTWARE.
 */

#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <stdbool.h>
#include <stdint.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <stdio.h>
#include <error.h>
#include <errno.h>
#include <time.h>
#include <sys/time.h>

#include <semaphore.h>
#include <assert.h>
#include <signal.h>

#include <xf86drm.h>
#include <xf86drmMode.h>
#include <libdrm/drm.h>

/* */
struct timeval tv_start;
struct timeval tv_end;

struct kms_display {
	drmModeConnector *connector;
	drmModeEncoder *encoder;
	drmModeModeInfo mode;
};

struct dumb_rb {
	uint32_t fb;
	uint32_t handle;
	uint32_t stride;
	uint64_t size;
	void *map;

	uint32_t connector_id;
	uint32_t h;
	uint32_t w;
	drmModeCrtcPtr saved_crtc;
};

struct dumb_rb dbo;

static bool find_drm_configuration(int fd, struct kms_display *kms,
				   uint32_t type)
{
	drmModeConnector *connector = NULL;
	drmModeEncoder *encoder = NULL;
	drmModeRes *resources = NULL;

	int i;

	resources = drmModeGetResources(fd);
	if (!resources) {
		fprintf(stderr, "drmModeGetResources failed\n");
		return false;
	}

	for (i = 0; i < resources->count_connectors; i++) {
		connector = drmModeGetConnector(fd, resources->connectors[i]);
		if (connector == NULL)
			continue;

		if (connector->connector_type != type)
			continue;
		if (connector->connection == DRM_MODE_CONNECTED &&
		    connector->count_modes > 0)
			break;

		drmModeFreeConnector(connector);
	}

	if (i == resources->count_connectors) {
		fprintf(stderr, "No currently active connector found\n");
		return false;
	}

	for (i = 0; i < resources->count_encoders; i++) {
		encoder = drmModeGetEncoder(fd, resources->encoders[i]);

		if (encoder == NULL)
			continue;

		if (encoder->encoder_id == connector->encoder_id)
			break;

		drmModeFreeEncoder(encoder);
	}

	if (i == resources->count_encoders) {
		fprintf(stderr, "No matching encoder for connector, "
			"use the first supported encoder\n");
		fprintf(stderr, "num encoders: %d\n",
			connector->count_encoders);
		if (connector->count_encoders)
			encoder = drmModeGetEncoder(fd, connector->encoders[0]);
		else {
			// FIXME: hard-coded value
			encoder = drmModeGetEncoder(fd, 3);
			encoder->crtc_id = 7;
		}

		if (!encoder) {
			fprintf(stderr, "Can't get preferred encoders\n");
			goto drm_free_connector;
		}
	}

	kms->connector = connector;
	kms->encoder = encoder;
	if (connector->count_modes) {
		kms->mode = connector->modes[0];
		printf("connector modes: %d\n",connector->count_modes);
		printf("pixel clock: %d\n", kms->mode.clock);
		printf("horizontal: %hd %hd %hd %hd %hd\n",
		       kms->mode.hdisplay, kms->mode.hsync_start,
		       kms->mode.hsync_end, kms->mode.htotal, kms->mode.hskew);
		printf("vertical: %hd %hd %hd %hd %hd\n",
		       kms->mode.vdisplay, kms->mode.vsync_start,
		       kms->mode.vsync_end, kms->mode.vtotal, kms->mode.vscan);
		printf("%d %d %d\n",
		       kms->mode.vrefresh, kms->mode.flags, kms->mode.type);
	} else {
			return false;
	}
    printf("MODE NAME: %s\n",kms->mode.name);

	return true;

drm_free_connector:
	drmModeFreeConnector(connector);

	return false;
}
static int set_crtc_properties(struct kms_display *kms_data, int fd)
{
	drmModeObjectPropertiesPtr props;
	drmModePropertyPtr prop;
	uint32_t i,ret = 0;

	props = drmModeObjectGetProperties(fd,
					   kms_data->encoder->crtc_id,
					   DRM_MODE_OBJECT_CRTC);

	for (i = 0; i < props->count_props; i++) {
		prop = drmModeGetProperty(fd, props->props[i]);

		if (!strcmp(prop->name, "alpha")) {
			ret = drmModeObjectSetProperty(
				fd, kms_data->encoder->crtc_id,
				DRM_MODE_OBJECT_CRTC, prop->prop_id,
				0);
			if (ret)
				perror("set global alpha failed");
		} else if (!strcmp(prop->name, "zpos")) {
			ret = drmModeObjectSetProperty(
				fd, kms_data->encoder->crtc_id,
				DRM_MODE_OBJECT_CRTC, prop->prop_id,
				1);
			if (ret)
				perror("set zpos failed");
		}

		drmModeFreeProperty(prop);
	}

	drmModeFreeObjectProperties(props);

	return ret;
}
static int dbo_configure(struct dumb_rb *dbo, int fd, uint32_t type)
{
	struct kms_display kms_data;
	struct dumb_rb dbo2 ;
	struct drm_mode_destroy_dumb dreq;
	struct drm_mode_create_dumb creq;
	struct drm_mode_create_dumb creq2;
	struct drm_mode_map_dumb mreq;
	int ret;

	memset(dbo, 0, sizeof(*dbo));
	memset(&dbo2, 0, sizeof(dbo2));
	memset(&mreq, 0, sizeof(mreq));
	memset(&creq, 0, sizeof(creq));
	memset(&creq2, 0, sizeof(creq2));
	memset(&dreq, 0, sizeof(dreq));


	/* find current DRM configuration */

	if (!find_drm_configuration(fd, &kms_data, type)) {
		fprintf(stderr, "failed to setup KMS\n");
		return -EFAULT;
	}

	/* create dumb buffer object */

	creq.height = kms_data.mode.vdisplay;
	creq.width = kms_data.mode.hdisplay;
	creq.bpp = 32;

	ret = drmIoctl(fd, DRM_IOCTL_MODE_CREATE_DUMB, &creq);
	if (ret) {
		fprintf(stderr,
			"failed drmIoctl(DRM_IOCTL_MODE_CREATE_DUMB)\n");
		return ret;
	}

	dbo->handle = creq.handle;
	dbo->stride = creq.pitch;
	dbo->size = creq.size;

	/* create framebuffer for dumb buffer object */

	ret = drmModeAddFB(fd, kms_data.mode.hdisplay, kms_data.mode.vdisplay,
			   32, 32, dbo->stride, dbo->handle, &dbo->fb);
	if (ret) {
		fprintf(stderr,
			"cannot add drm framebuffer for dumb buffer object\n");
		return ret;
	}

	/* create dumb buffer 2 object */

	creq2.height = kms_data.mode.vdisplay;
	creq2.width = kms_data.mode.hdisplay;
	creq2.bpp = 32;

	ret = drmIoctl(fd, DRM_IOCTL_MODE_CREATE_DUMB, &creq2);
	if (ret) {
		fprintf(stderr, "failed drmIoctl(DRM_IOCTL_MODE_CREATE_DUMB) for 2\n");
		return ret;
	}

	dbo2.handle = creq2.handle;
	dbo2.stride = creq2.pitch;
	dbo2.size = creq2.size;

	/* create framebuffer 2 for dumb buffer 2 object */

	ret = drmModeAddFB(fd, kms_data.mode.hdisplay, kms_data.mode.vdisplay,
			   32, 32, dbo2.stride, dbo2.handle, &dbo2.fb);
	if (ret) {
		fprintf(stderr, "cannot add drm framebuffer 2 for dumb buffer 2 object\n");
		return ret;
	}

	/* map dumb buffer object */

	mreq.handle = dbo->handle;

	ret = drmIoctl(fd, DRM_IOCTL_MODE_MAP_DUMB, &mreq);
	if (ret) {
		fprintf(stderr, "failed drmIoctl(DRM_IOCTL_MODE_MAP_DUMB)\n");
		return ret;
	}

	dbo->map = mmap(0, dbo->size, PROT_READ | PROT_WRITE, MAP_SHARED,
			fd, mreq.offset);
	if (dbo->map == MAP_FAILED) {
		fprintf(stderr, "cannot mmap dumb buffer\n");
		return -EFAULT;
	}
	memset(dbo->map,0,dbo->size);
	/* store current crtc */

	dbo->saved_crtc = drmModeGetCrtc(fd, kms_data.encoder->crtc_id);
	if (dbo->saved_crtc == NULL) {
		fprintf(stderr, "failed to get current mode\n");
	}

	/* setup new crtc */
	printf("SET dbo1: encoder %d, crtc %d, connector %d\n",
	       kms_data.encoder->encoder_id, kms_data.encoder->crtc_id,
	       kms_data.connector->connector_id);
	gettimeofday(&tv_start, NULL);

	printf("Before drmModeSetCtrc: %dx%d\n",kms_data.mode.hdisplay,kms_data.mode.vdisplay);
	ret = drmModeSetCrtc(fd, kms_data.encoder->crtc_id, dbo->fb, 0, 0,
			     &kms_data.connector->connector_id, 1, &kms_data.mode);
	if (ret) {
		fprintf(stderr, "cannot set new drm crtc for 1 again\n");
		return ret;
	}

	dbo->h = kms_data.mode.vdisplay;
	dbo->w = kms_data.mode.hdisplay;
	dbo->connector_id = kms_data.connector->connector_id;

	ret = set_crtc_properties(&kms_data, fd);
	if (ret < 0) {
        	fprintf(stderr, "cannot set properties of the crtc\n");
		return ret;
	}

	printf("After drmModeSetCtrc: %dx%d\n",kms_data.mode.hdisplay,kms_data.mode.vdisplay);

	return 0;
}

static void dbo_destroy(struct dumb_rb *dbo, int fd)
{
	struct drm_mode_destroy_dumb dreq;
	int ret;

	if (dbo->saved_crtc) {
		ret = drmModeSetCrtc(fd, dbo->saved_crtc->crtc_id,
				     dbo->saved_crtc->buffer_id,
				     dbo->saved_crtc->x, dbo->saved_crtc->y,
				     &dbo->connector_id, 1,
				     &dbo->saved_crtc->mode);
		if (ret) {
			fprintf(stderr, "cannot restore old CRTC\n");
		}
	}

	if (dbo->map)
		munmap(dbo->map, dbo->size);

	drmModeRmFB(fd, dbo->fb);

	dreq.handle = dbo->handle;
	ret = drmIoctl(fd, DRM_IOCTL_MODE_DESTROY_DUMB, &dreq);
	if (ret) {
		fprintf(stderr, "cannot destroy dumb buffer\n");
	}
}

#if 0
static void do_draw(struct dumb_rb *dbo)
{
	uint32_t color32[] = {
		0xFF000000, 0x000000FF, 0xFF00FF00, 0x0000FFFF,
		0xFFFF0000, 0x00FF00FF, 0xFFFFFF00, 0x00FFFFFF,
	};

	uint32_t *dst = dbo->map;
	uint32_t h = dbo->h;
	uint32_t w = dbo->w;
	uint32_t color, i, j;

	printf("2: %dx%d\n", w, h);
	for(i = 0; i < h; i++ ){

		color = color32[8*i/h];

		for(j = 0; j < w; j++) {
			*(dst + i*w + j) = color;
		}
	}
}
#endif

/* */
static const char * connector_type[] = {
	[DRM_MODE_CONNECTOR_Unknown] =     "Unknown",
	[DRM_MODE_CONNECTOR_VGA] =         "VGA",
	[DRM_MODE_CONNECTOR_DVII] =        "DVII",
	[DRM_MODE_CONNECTOR_DVID] =        "DVID",
	[DRM_MODE_CONNECTOR_DVIA] =        "DVIA",
	[DRM_MODE_CONNECTOR_Composite] =   "Composite",
	[DRM_MODE_CONNECTOR_SVIDEO] =      "SVIDEO",
	[DRM_MODE_CONNECTOR_LVDS] =        "LVDS",
	[DRM_MODE_CONNECTOR_Component] =   "Component",
	[DRM_MODE_CONNECTOR_9PinDIN] =     "9PinDIN",
	[DRM_MODE_CONNECTOR_DisplayPort] = "DisplayPort",
	[DRM_MODE_CONNECTOR_HDMIA] =       "HDMIA",
	[DRM_MODE_CONNECTOR_HDMIB] =       "HDMIB",
	[DRM_MODE_CONNECTOR_TV] =          "TV",
	[DRM_MODE_CONNECTOR_eDP] =         "eDP",
};
#define NUM_CONNECTOR_TYPES (DRM_MODE_CONNECTOR_eDP + 1)

void drm_mode_set_helper_terminate(int fd_drm)
{
	dbo_destroy(&dbo, fd_drm);
}

int drm_mode_set_helper_init(char *connector_name, int fd_drm)
{
	uint64_t has_dumb;
	uint32_t type;
	int ret;

	for (type = 0; type < NUM_CONNECTOR_TYPES; type++)
		if (!strcasecmp(connector_name, connector_type[type]))
			break;
	if (type >= NUM_CONNECTOR_TYPES) {
		fprintf(stderr, "Unknown connector type %s", connector_name);
		exit(1);
	}

	drmSetMaster(fd_drm);

	if (drmGetCap(fd_drm, DRM_CAP_DUMB_BUFFER, &has_dumb) < 0) {
		perror("DRM_CAP_DUMB_BUFFER ioctl");
		ret = -EFAULT;
		goto err_close;
	}

	if (!has_dumb) {
		fprintf(stderr, "driver does not support dumb buffers\n");
		ret = -EFAULT;
		goto err_close;
	}

	if (dbo_configure(&dbo, fd_drm, type)) {
		goto err_unmap;
	}
	drmDropMaster(fd_drm);

#if 0
	do_draw(&dbo);
#endif

	return 0;

err_unmap:
	dbo_destroy(&dbo, fd_drm);
err_close:
	return ret;
}
